<?php

/**
 * This is the model class for table "Kontrahent".
 *
 * The followings are the available columns in table 'Kontrahent':
 * @property double $id
 * @property string $nazwa
 * @property string $urlLoga
 * @property string $NIP
 * @property string $REGON
 * @property string $nrKonta
 * @property string $idBudynku
 * @property string $adres
 * @property int    $typ
 * @property int    $idDziedziny
 *
 * @property Dziedzina $dziedzina
 * @property Budynek   $budynek
 * @property Oplaty[]  $oplaties
 * @property Faktury[] $fakturies
 */
class Kontrahent extends CActiveRecord
{
	const TYPE_STANDARD = 1;
	const TYPE_CENTER = 2;
	const TYPE_FOREIGN = 3;
	const TYPE_ADAGENCY = 4;
	const TYPE_PLACEMENTAGENCY = 5;
	const TYPE_BANK = 6;
	const TYPE_ZUS = 7;
	const TYPE_US = 8;

	/**
	 * Returns the static model of the specified AR class.
	 * @param string $className active record class name.
	 * @return Kontrahent the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return 'Kontrahent';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public function rules()
	{
		return array(
			array('nazwa, adres, NIP, REGON, nrKonta, typ, idDziedziny', 'required'),
			array('typ', 'default', 'value' => self::TYPE_STANDARD),
			array('typ, idDziedziny', 'numerical'),
			array('nazwa', 'length', 'max'=>128),
			array('urlLoga', 'length', 'max'=>512),
			array('adres', 'length', 'max'=>200),
			array('NIP', 'length', 'max'=>20),
			array('REGON', 'length', 'max'=>18),
			array('nrKonta', 'length', 'max'=>64),
			array('NIP', 'ext.Validators.NipValidator'),
			array('REGON', 'ext.Validators.Regon9Validator'),
			array('id, nazwa, urlLoga, NIP, REGON, adres, idBudynku, typ, idDziedziny', 'safe', 'on'=>'search'),
		);
	}

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		return array(
			'fakturies'=>array(self::HAS_MANY, 'Faktury', 'idKontrahenta'),
			'oplaties'=>array(self::HAS_MANY, 'Oplaty', 'idKontrahenta'),
			'budynek'=>array(self::BELONGS_TO, 'Budynek', 'idBudynku'),
			'dziedzina'=>array(self::BELONGS_TO, 'Dziedzina', 'idDziedziny'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id'=>Yii::t('Kontrahent', 'ID'),
			'nazwa'=>Yii::t('Kontrahent', 'Nazwa'),
			'urlLoga'=>Yii::t('Kontrahent', 'Logo'),
			'NIP'=>Yii::t('Kontrahent', 'NIP'),
			'REGON'=>Yii::t('Kontrahent', 'REGON'),
			'adres'=>Yii::t('Kontrahent', 'Adres'),
			'nrKonta'=>Yii::t('Kontrahent', 'Numer konta'),
			'idBudynku'=>Yii::t('Kontrahent', 'Budynek'),
			'typ'=>Yii::t('Kontrahent', 'Rodzaj'),
			'idDziedziny'=>Yii::t('Kontrahent', 'Dziedzina'),
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 * @return CActiveDataProvider the data provider that can return the models based on the search/filter conditions.
	 */
	public function search()
	{
		$criteria = new CDbCriteria;

		$criteria->compare('"t"."id"', $this->id);
		$criteria->compare('LOWER("t"."nazwa")', strtolower($this->nazwa), true);
		$criteria->compare('LOWER("t"."urlLoga")', strtolower($this->urlLoga), true);
		$criteria->compare('LOWER("t"."NIP")', strtolower($this->NIP), true);
		$criteria->compare('LOWER("t"."REGON")', strtolower($this->REGON), true);
		$criteria->compare('LOWER("t"."nrKonta")', strtolower($this->nrKonta), true);
		$criteria->compare('LOWER("t"."idBudynku")', $this->idBudynku);
		$criteria->compare('"t"."typ"', $this->typ);
		$criteria->compare('"t"."idDziedziny"', $this->idDziedziny);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	static private $_kontrahenci;

	public static function listAll()
	{
		if (empty(self::$_kontrahenci))
			self::$_kontrahenci = CHtml::listData (self::model()->findAll(), 'id', 'nazwa');
		return self::$_kontrahenci;
	}

	private static $types = array(
		self::TYPE_STANDARD => 'Standardowy',
		self::TYPE_CENTER => 'Centrum szkoleniowe',
		self::TYPE_FOREIGN => 'Zagraniczny',
		self::TYPE_ADAGENCY => 'Agencja reklamowa',
		self::TYPE_PLACEMENTAGENCY => 'Agencja pośrednictwa pracy',
		self::TYPE_BANK => 'Bank',
		self::TYPE_ZUS => 'ZUS',
		self::TYPE_US => 'Urząd skarbowy'
	);

	public static function getTypeStrArr()
	{
		return self::$types;
	}

	public function getTypStr()
	{
		return self::$types[$this->typ];
	}

	public static function getReputation($companyId, $kontrahentId)
	{
		$ratio = Yii::app()->db->createCommand('
			select
				sum(nvl2("t"."idOdpowiedzi", 1, 0)) / count(*) "ratio"
			from "ZlecenieFirmy" "t"
			where "t"."idFirmy" = :idFirmy and "t"."idKontrahenta" = :kontrahentId and "t"."czasZakonczenia" is not null
		')->queryScalar(array(
			':kontrahentId' => $kontrahentId,
			':idFirmy' => $companyId
		));

		// jeżeli $ratio jako int jest co do wartości równe $ratio jako float i $ratio jako int jest równe 0,
		// to znaczy, że wyliczona reputacja jest 0 i wtedy ustawiamy jakąś bardzo małą reputację, żeby
		// ominąć przypadek, gdy dzięki temu, że reputacja jest 0 to tak jakbyśmy w ogóle dla kontrahenta nie robili jeszcze zleceń
		if($ratio !== null)
			$ratio = (int)$ratio == (float)$ratio && (int)$ratio == 0 ? '0.0001' : $ratio;

		return $ratio;
	}

	public function canCooperate($companyId)
	{
		return self::companiesCanCooperate($companyId, $this->id);
	}

	public static function companiesCanCooperate($companyId, $kontrahentId)
	{
		$ratio = self::getReputation($companyId, $kontrahentId);
		return $ratio == null ? true : $ratio >= (float)ParametryGry::getValue('ZLECENIE_KONTRAHENT_PROG', 0.5);
	}

	public static function isTenderAvileable($kontrahentId, $actionId = null)
	{
		if (empty($actionId))
			$actionId = Akcja::getLatestTenderActionId(Yii::app()->user);
		$tenders = (int)Yii::app()->db->createCommand('
			select count(distinct p."id") from "Przetarg" p
			left join "PrzetargGrupa" pg on pg."idPrzetargu" = p."id"
			left join "PrzetargFirma" pf on pf."idGrupy" = pg."id" and pf."idFirmy" = :companyId
			where pf."id" is null and p."idAkcji" = :idAkcji and p."idKlasy" = :idKlasy
				and p."idKontrahenta" = :idKontrahenta
		')->queryScalar(array(
			':idAkcji' => $actionId,
			':idKlasy' => Yii::app()->user->id_klasy,
			':idKontrahenta' => $kontrahentId,
			':companyId' => Yii::app()->user->id_firmy
		));
		return $tenders;
	}
}
